#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "build.h"
#include "display.h"


// Amiga includes.
#include <proto/exec.h>
#include <proto/graphics.h>
#include <proto/intuition.h>
#include <proto/lowlevel.h>
#include <proto/dos.h>
#include <proto/timer.h>

#include <clib/icon_protos.h>
#include <workbench/startup.h>


#include "amiga_c2p_aga.h"
#include "amiga_keyboard_codes.h"
#include "keyboard_scan_codes.h"







char *ID = "$VER: AmiDuke 1.10\r\n";  

#define MAX_ARGVS   100
char *_argv[MAX_ARGVS];
int _argc = 0;

static char systemPath[MAX_PATH];


long bytesperline, frameplace;
char *screen;
long buffermode, origbuffermode, linearmode;


typedef struct AMIGA_ScreenRes
{
    int w,h;
} AMIGA_ScreenRes;

typedef struct AMIGA_Color
{
   unsigned char r,g,b;
} AMIGA_Color;


#define NUM_SCREEN_MODES    4
static AMIGA_ScreenRes videoModes[NUM_SCREEN_MODES];


static struct Screen *_hardwareScreen = NULL;
static struct Window *_hardwareWindow = NULL;

// Hardware double buffering.
static struct ScreenBuffer *_hardwareScreenBuffer[2];
static BYTE _currentScreenBuffer = 0;

// AGA C2P.
static void *c2p[2] = {NULL, NULL};

static int closeWorkbench = 0;
static int workbenchClosed = 0;

static ULONG monitorType = PAL_MONITOR_ID;

#define AGA_VIDEO_DEPTH 8

#define AGA_MAX_PAL_VIDEO_WIDTH 640
#define AGA_MAX_PAL_VIDEO_HEIGHT    512

#define AGA_MAX_NTSC_VIDEO_WIDTH 640
#define AGA_MAX_NTSC_VIDEO_HEIGHT    400



static UWORD _emptypointer[] = {
  0x0000, 0x0000,    /* reserved, must be NULL */
  0x0000, 0x0000,     /* 1 row of image data */
  0x0000, 0x0000    /* reserved, must be NULL */
};

static int *timerinthandle = NULL;


static unsigned char scancodes[256];

static int mouse_relative_x = 0;
static int mouse_relative_y = 0;
static short mouse_buttons = 0;




static unsigned int __interrupt _timer_catcher(void *bleh, void *bluh)
{
    totalclock++;

    return 0;
}

static void AMIGA_StartTimer(unsigned int intervall)
{
    timerinthandle=AddTimerInt((APTR)_timer_catcher, NULL);
    if (timerinthandle)
    {
	   StartTimerInt(timerinthandle,intervall*1000,TRUE);
    }
}

static void AMIGA_EndTimer(void)
{
    if (timerinthandle)
    {
    	StopTimerInt(timerinthandle);
    	RemTimerInt(timerinthandle);
    	timerinthandle=NULL;
    }
}

char *_getSystemPath(void)
{
    return systemPath;
}



void AMIGA_CloseDisplay(void)
{
    if (_hardwareWindow) {
        ClearPointer(_hardwareWindow);
        CloseWindow(_hardwareWindow);
        _hardwareWindow = NULL;
	}

	if (_hardwareScreenBuffer[0]) {
        ChangeScreenBuffer (_hardwareScreen, _hardwareScreenBuffer[0]);
        WaitTOF();
        WaitTOF();
        FreeScreenBuffer (_hardwareScreen, _hardwareScreenBuffer[0]);
        _hardwareScreenBuffer[0] = NULL;
    }

    if (_hardwareScreenBuffer[1]) {
        FreeScreenBuffer (_hardwareScreen, _hardwareScreenBuffer[1]);
        _hardwareScreenBuffer[1] = NULL;
    }

	if (_hardwareScreen) {
        CloseScreen(_hardwareScreen);
        _hardwareScreen = NULL;
    }

    if (c2p[0]) {
        c2p8_deinit_stub(c2p[0]);
        c2p[0] = NULL;
    }

    if (c2p[1]) {
        c2p8_deinit_stub(c2p[1]);
        c2p[1] = NULL;
    }
}









/*
 * !!! This is almost an entire copy of the original setgamemode().
 * !!!  Figure out what is needed for just 2D mode, and separate that
 * !!!  out. Then, place the original setgamemode() back into engine.c,
 * !!!  and remove our simple implementation (and this function.)
 * !!!  Just be sure to keep the non-DOS things, like the window's
 * !!!  titlebar caption.   --ryan.
 */
static char screenalloctype = 255;
static void init_new_res_vars(int davidoption)
{
    int i = 0;
    int j = 0;

    bytesperline = xdim;


    linearmode = 1;
    activepage = visualpage = 0;
    horizlookup = horizlookup2 = NULL;

    frameplace = NULL;

    if (screen != NULL)
    {
        if (screenalloctype == 0) kkfree((void *)screen);
        if (screenalloctype == 1) suckcache((long *)screen);

        screen = NULL;
    }

    if (davidoption != -1)
    {
    	switch(vidoption)
    	{
    		case 1:i = xdim*ydim; break;
    		case 2: xdim = 320; ydim = 200; i = xdim*ydim; break;
    		default: assert(0);
    	}

    	j = ydim*4*sizeof(long);  /* Leave room for horizlookup&horizlookup2 */

    	screenalloctype = 0;
    	if ((screen = (char *)kkmalloc(i+(j<<1))) == NULL)
    	{
    		 allocache((long *)&screen,i+(j<<1),&permanentlock);
    		 screenalloctype = 1;
    	}

    	frameplace = FP_OFF(screen);
    	horizlookup = (long *)(frameplace+i);
    	horizlookup2 = (long *)(frameplace+i+j);
    }

    j = 0;
    for(i = 0; i <= ydim; i++)
    {
    	ylookup[i] = j;
    	j += bytesperline;
    }

    horizycent = ((ydim*4)>>1);

    /* Force drawrooms to call dosetaspect & recalculate stuff */
    oxyaspect = oxdimen = oviewingrange = -1;

    setvlinebpl(bytesperline);

    if (davidoption != -1)
    {
    	setview(0L,0L,xdim-1,ydim-1);
    	clearallviews(0L);
    }

    setbrightness((char) curbrightness, (unsigned char *) &palette[0]);

    if (searchx < 0) { searchx = halfxdimen; searchy = (ydimen>>1); }
    
    // Reset.
     mouse_relative_x = mouse_relative_y = 0;
     mouse_buttons = 0;
}

static int AMIGA_SetVideoMode(int width, int height)
{
	ULONG modeId = INVALID_ID;
	DisplayInfoHandle handle;
	struct DisplayInfo dispinfo;
	struct DimensionInfo dimsinfo;
	
    AMIGA_LogMessage("Setting video mode %dx%d.\n", width, width);

	// Check args.
    if (monitorType == PAL_MONITOR_ID) {
       	if (width > AGA_MAX_PAL_VIDEO_WIDTH) {
            AMIGA_LogWarning("Width cannot be greater than %d pixels\n", AGA_MAX_PAL_VIDEO_WIDTH);
            return 0;
        }
    
    	if (height > AGA_MAX_PAL_VIDEO_HEIGHT) {
            AMIGA_LogWarning("Height cannot be greater than %d pixels\n", AGA_MAX_PAL_VIDEO_HEIGHT);
            return 0;
        }
    } else {
        	if (width > AGA_MAX_NTSC_VIDEO_WIDTH) {
            AMIGA_LogWarning("Width cannot be greater than %d pixels\n", AGA_MAX_NTSC_VIDEO_WIDTH);
            return 0;
        }

    	if (height > AGA_MAX_NTSC_VIDEO_HEIGHT) {
            AMIGA_LogWarning("Height cannot be greater than %d pixels\n", AGA_MAX_NTSC_VIDEO_HEIGHT);
            return 0;
        }
    }

    // Free any existing mode.
    AMIGA_CloseDisplay();

	// Automatically choose the best mode.
	modeId = BestModeID(BIDTAG_NominalWidth, width,
					BIDTAG_NominalHeight, height,
					BIDTAG_DesiredWidth, width,
					BIDTAG_DesiredHeight, height,
					BIDTAG_Depth, AGA_VIDEO_DEPTH,
					BIDTAG_MonitorID, monitorType,
					TAG_END);

	// Verify the mode choosen.
	if (modeId != INVALID_ID) {
		if ((handle = FindDisplayInfo(modeId)) == NULL) {
			AMIGA_LogWarning("Couldn't find Display Info for requested mode\n");
			return 0;
		}

		if (GetDisplayInfoData(handle, (UBYTE *)&dispinfo, sizeof(dispinfo), DTAG_DISP,0) == 0) {
			AMIGA_LogWarning("Couldn't get Display Info Data for requested mode\n");
			return 0;
		}

		if (GetDisplayInfoData(handle, (UBYTE *)&dimsinfo, sizeof(dimsinfo), DTAG_DIMS, 0) == 0) {
			AMIGA_LogWarning("Couldn't get Display Info Data for requested mode\n");
			return 0;
		}

		if (dimsinfo.MaxDepth != AGA_VIDEO_DEPTH) {
		   modeId = INVALID_ID;
		}

		if ((dimsinfo.Nominal.MaxX + 1) != width) {
		   modeId = INVALID_ID;
		}

		if ((dimsinfo.Nominal.MaxY + 1) < height) {
		   modeId = INVALID_ID;
		}
	}

	if (modeId == INVALID_ID) {
		AMIGA_LogWarning("Couldn't find a Screen Mode for requested mode\n");
		return 0;
    }


     // Create the hardware screen.
	_hardwareScreen = OpenScreenTags(NULL,
                     SA_Depth, AGA_VIDEO_DEPTH,
                     SA_DisplayID, modeId,
					 SA_Top, 0,
					 SA_Left, 0,
                     SA_Height, height,
                     SA_Width, width,
					 SA_Height, height,
					 SA_Type, CUSTOMSCREEN,
                     SA_Quiet, TRUE,
					 SA_ShowTitle, FALSE,
					 SA_Draggable, FALSE,
                     SA_Exclusive, TRUE,
					 SA_AutoScroll, FALSE,
                     TAG_END);

	if (!_hardwareScreen ) {
		AMIGA_LogWarning("Couldn't create a Hardware Screen for requested mode\n");
		return 0;
	}

    // Setup double buffering.
	_hardwareScreenBuffer[0] = AllocScreenBuffer (_hardwareScreen, NULL, SB_SCREEN_BITMAP);
	_hardwareScreenBuffer[1] = AllocScreenBuffer (_hardwareScreen, NULL, 0);

	c2p[0] = c2p8_reloc_stub(_hardwareScreenBuffer[0]->sb_BitMap);
	c2p[1] = c2p8_reloc_stub(_hardwareScreenBuffer[1]->sb_BitMap);

    _currentScreenBuffer = 1;

    _hardwareWindow = OpenWindowTags(NULL,
              	    WA_Left, 0,
        			WA_Top, 0,
        			WA_Width, width,
        			WA_Height, height,
        			WA_CustomScreen, (ULONG)_hardwareScreen,
        			WA_Backdrop, TRUE,
        			WA_Borderless, TRUE,
        			WA_Activate, TRUE,
        			WA_SimpleRefresh, TRUE,
        			WA_NoCareRefresh, TRUE,
        			WA_ReportMouse, TRUE,
        			WA_RMBTrap, TRUE,
              	    WA_IDCMP, IDCMP_RAWKEY|IDCMP_MOUSEMOVE|IDCMP_DELTAMOVE|IDCMP_MOUSEBUTTONS,
              	    TAG_END);

	if (!_hardwareWindow) {
		AMIGA_LogWarning("Couldn't create a Hardware Window for requested mode\n");
		return 0;
	}
	
    // Hide WB mouse pointer.
	SetPointer(_hardwareWindow, _emptypointer, 1, 16, 0, 0);
	
	// Only close the WB once!
    if ((closeWorkbench) && (!workbenchClosed)) {
        AMIGA_LogMessage("Closing the Workbench\n");
        workbenchClosed = CloseWorkBench();
    }

    
    xdim = width;
    ydim = height;    
    
    // All is cool!
    return 1;
}

static void add_vesa_mode(const char *typestr, int w, int h)
{
    AMIGA_LogMessage("Adding %s resolution (%dx%d)\n", typestr, w, h);
    validmode[validmodecnt] = validmodecnt;
    validmodexdim[validmodecnt] = w;
    validmodeydim[validmodecnt] = h;
    validmodecnt++;
}


static void remove_vesa_mode(int index, const char *reason)
{
    int i;

    assert(index < validmodecnt);
    AMIGA_LogMessage("Removing resolution #%d, %dx%d [%s]\n", index, validmodexdim[index], validmodeydim[index], reason);

    for (i = index; i < validmodecnt - 1; i++)
    {
    	validmode[i] = validmode[i + 1];
    	validmodexdim[i] = validmodexdim[i + 1];
    	validmodeydim[i] = validmodeydim[i + 1];
    }

    validmodecnt--;
}

static void cull_large_vesa_modes(void)
{
    long max_w = MAX_DISPLAY_WIDTH;
    long max_h = MAX_DISPLAY_HEIGHT;
    int i;

    for (i = 0; i < validmodecnt; i++)
    {
    	if ((validmodexdim[i] > max_w) || (validmodeydim[i] > max_h))
    	{
    	    remove_vesa_mode(i, "above resolution ceiling");
    	    i--;  /* list shrinks. */
    	}
    }
}

static void cull_duplicate_vesa_modes(void)
{
    int i;
    int j;

    for (i = 0; i < validmodecnt; i++)
    {
    	for (j = i + 1; j < validmodecnt; j++)
    	{
    	    if ( (validmodexdim[i] == validmodexdim[j]) && (validmodeydim[i] == validmodeydim[j]) )
    	    {
        		remove_vesa_mode(j, "duplicate");
        		j--;  /* list shrinks. */
    	    }
    	}
    }
}


#define swap_macro(tmp, x, y) { tmp = x; x = y; y = tmp; }

/* be sure to call cull_duplicate_vesa_modes() before calling this. */
static void sort_vesa_modelist(void)
{
    int i;
    int sorted;
    long tmp;

    do
    {
    	sorted = 1;
    	for (i = 0; i < validmodecnt - 1; i++)
    	{
    	    if ( (validmodexdim[i] >= validmodexdim[i+1]) && (validmodeydim[i] >= validmodeydim[i+1]) )
    	    {
        		sorted = 0;
        		swap_macro(tmp, validmode[i], validmode[i+1]);
        		swap_macro(tmp, validmodexdim[i], validmodexdim[i+1]);
        		swap_macro(tmp, validmodeydim[i], validmodeydim[i+1]);
    	    }
    	}
    } while (!sorted);
}

static void cleanup_vesa_modelist(void)
{
    cull_large_vesa_modes();
    cull_duplicate_vesa_modes();
    sort_vesa_modelist();
}

static void output_vesa_modelist(void)
{
    char buffer[256];
    char numbuf[20];
    int i;

    buffer[0] = '\0';

    for (i = 0; i < validmodecnt; i++)
    {
	sprintf(numbuf, " (%ldx%ld)",
		  (long) validmodexdim[i], (long) validmodeydim[i]);

	if ( (strlen(buffer) + strlen(numbuf)) >= (sizeof (buffer) - 1) )
	    strcpy(buffer + (sizeof (buffer) - 5), " ...");
	else
	    strcat(buffer, numbuf);
    }
}

static AMIGA_ScreenRes *AMIGA_GetResolutions(int *nummodes)
{
    *nummodes = NUM_SCREEN_MODES;

    return &videoModes[0];
}

void getvalidvesamodes(void)
{
    static int already_checked = 0;
    int i, nummodes;
    AMIGA_ScreenRes *modes = NULL;


    if (already_checked)
	   return;

    already_checked = 1;
    validmodecnt = 0;
    vidoption = 1;  /* !!! tmp */


	/* Anything the hardware can do is added now... */
    modes = AMIGA_GetResolutions(&nummodes);

    for (i = 0; i<nummodes; i++)
	   add_vesa_mode("physical", modes[i].w, modes[i].h);

	/* get rid of dupes and bogus resolutions... */
    cleanup_vesa_modelist();

	/* print it out for debugging purposes... */
    output_vesa_modelist();
}

int _setgamemode(int davidoption, long daxdim, long daydim)
{
    getvalidvesamodes();

    
    if (!AMIGA_SetVideoMode(daxdim, daydim))
    {
    	return 0;
    } 

    init_new_res_vars(davidoption);

    // All good.
    return 1;
}























static void AMIGA_SetColors(AMIGA_Color *cols, int start, int num)
{
    int i;
    unsigned int colortable[780];

    colortable[0] = (num<<16)+start;
    for( i=0; i<num; i++ )
    {
    	colortable[i*3+1]=cols[i].r<<24;
    	colortable[i*3+2]=cols[i].g<<24;
    	colortable[i*3+3]=cols[i].b<<24;
    }
    colortable[num*3+1]=0;

    LoadRGB32(&_hardwareScreen->ViewPort, (ULONG*)colortable );
}

int VBE_setPalette(long start, long num, char *palettebuffer)
/*
 * (From Ken's docs:)
 *   Set (num) palette palette entries starting at (start)
 *   palette entries are in a 4-uint8 format in this order:
 *       0: Blue (0-63)
 *       1: Green (0-63)
 *       2: Red (0-63)
 *       3: Reserved
 *
 * Naturally, the bytes are in the reverse order that SDL wants them...
 *  More importantly, SDL wants the color elements in a range from 0-255,
 *  so we do a conversion.
 */
{
    AMIGA_Color fmt_swap[256];
    AMIGA_Color *sdlp = &fmt_swap[start];
    char *p = palettebuffer;
    int i;

    for (i = 0; i < num; i++)
    {
    	sdlp->b = (unsigned char) ((((float) *p++) / 63.0) * 255.0);
    	sdlp->g = (unsigned char) ((((float) *p++) / 63.0) * 255.0);
    	sdlp->r = (unsigned char) ((((float) *p++) / 63.0) * 255.0);
    	p++;

    	sdlp++;
    }

    AMIGA_SetColors(fmt_swap, start, num);

    return 1;
}

static void AMIGA_GetColors(AMIGA_Color *cols)
{
    int i;
    unsigned int colortable[768];

    GetRGB32(_hardwareScreen->ViewPort.ColorMap,0,256,(ULONG*)colortable);

    for( i=0; i<256; i++ )
    {
    	cols[i].r = colortable[i*3]>>24;
    	cols[i].g = colortable[i*3+1]>>24;
    	cols[i].b = colortable[i*3+2]>>24;
    }
}

int VBE_getPalette(long start, long num, char *palettebuffer)
{
    AMIGA_Color sdlp[256];
    char *p = palettebuffer + (start * 4);
    int i;

    AMIGA_GetColors(sdlp);

    for (i = 0; i < num; i++)
    {
    	*p++ = (unsigned char) ((((float) sdlp[start+i].b) / 255.0) * 63.0);
    	*p++ = (unsigned char) ((((float) sdlp[start+i].g) / 255.0) * 63.0);
    	*p++ = (unsigned char) ((((float) sdlp[start+i].r) / 255.0) * 63.0);
    	*p++ = 0;
    }

    return(1);
}














int screencapture(char *filename, char inverseit)
{
    // TODO
}




void _nextpage(void)
{
    c2p8_stub(c2p[_currentScreenBuffer], _hardwareScreenBuffer[_currentScreenBuffer]->sb_BitMap, (unsigned char*)frameplace, (xdim * ydim));

    if (ChangeScreenBuffer(_hardwareScreen, _hardwareScreenBuffer[_currentScreenBuffer])) {
        // Flip.
	   _currentScreenBuffer = _currentScreenBuffer ^ 1;
    }
}





static inline void amiga_key_filter(unsigned char code, int pressed)
{
    unsigned char rawkey;

    rawkey = scancodes[code];
    if (rawkey != sc_None) {
	   KB_KeyEvent(rawkey, pressed);;
    }
}

void _handle_events(void)
{
    struct IntuiMessage *msg;
    unsigned char code;
    int qual;

    
    while((msg = (struct IntuiMessage *)GetMsg(_hardwareWindow->UserPort)))
    {
        // Extract data and reply message
    	code = msg->Code;
    	qual = msg->Qualifier;

    	// Action depends on message class
    	switch (msg->Class)
    	{
    	    case IDCMP_RAWKEY:
    		if (code & IECODE_UP_PREFIX )
    		{
    		    code &= 0x7f;
    		    amiga_key_filter(code, FALSE);
    		}
    		else
    		{
    		    amiga_key_filter(code, TRUE);
    		}
    	    break;

    	    case IDCMP_MOUSEBUTTONS:
        		switch( code )
        		{
        		    case SELECTDOWN:
        			mouse_buttons |= 0x01;
        		    break;
    
        		    case MIDDLEDOWN:
        			mouse_buttons |= 0x04;
        		    break;
    
        		    case MENUDOWN:
        			mouse_buttons |= 0x02;
        		    break;
    
        		    case SELECTUP:
        			mouse_buttons &= 0xfe;
        		    break;
    
        		    case MIDDLEUP:
        			mouse_buttons &= 0xfb;
        		    break;
    
        		    case MENUUP:
        			mouse_buttons &= 0xfd;
        		    break;
        		}
    	    break;

    	    case IDCMP_MOUSEMOVE:
        		mouse_relative_x = msg->MouseX;
                mouse_relative_y = msg->MouseY;
        	    break;
    	}
    	
    	ReplyMsg((struct Message *)msg);
    }
}




void _getMousePosition(int *x, int *y)
{
    *x = mouse_relative_x;
    *y = mouse_relative_y;

    // Reset.
    mouse_relative_x = mouse_relative_y = 0;
}


short _getMouseButtons(void)
{
    // special wheel treatment
    //if(mouse_buttons&8) mouse_buttons ^= 8;
    //if(mouse_buttons&16) mouse_buttons ^= 16;

    return mouse_buttons;
}




void _idle(void)
{
	_handle_events();
}



int _platform_init(int argc, char **argv)
{
    char path[MAX_PATH];
    struct WBStartup* wbStartup;
    struct DiskObject *diskObject;
    char *toolType;
    
    AMIGA_LogInit();


    if (argc != 0) {
        // Started from the shell.
       return -1;
    }

    // Setup command line.
    wbStartup = (struct WBStartup*)argv;

    NameFromLock(wbStartup->sm_ArgList[0].wa_Lock, path, MAX_PATH);
    CurrentDir(wbStartup->sm_ArgList[0].wa_Lock);

    // Fix up path.
    sprintf(systemPath, "%s/", path);


    // Setup command line arguments.
    _argv[_argc] = (char *)kmalloc(strlen("AmiDuke")+1);
    strcpy(_argv[_argc++], "AmiDuke");

    // Process Tooltypes.
    diskObject = GetDiskObject((char*)wbStartup->sm_ArgList[0].wa_Name);
    if (diskObject != NULL) {
		toolType = (char*)FindToolType(diskObject->do_ToolTypes, "CLOSE_WB");
        if (toolType != NULL) {
            closeWorkbench = 1;
        }

        // This won't be used by RTG but won't do any harm.
        toolType = (char*)FindToolType(diskObject->do_ToolTypes, "DISPLAY_MODE");
        if (toolType != NULL) {
            if (strcmp(toolType, "NTSC") == 0) {
                monitorType = NTSC_MONITOR_ID;
            }
        }
    }


    memset(scancodes, sc_None, sizeof (scancodes));
    scancodes[RAWKEY_ESCAPE]          = sc_Escape;
    scancodes[RAWKEY_F1]              = sc_F1;
    scancodes[RAWKEY_F2]              = sc_F2;
    scancodes[RAWKEY_F3]              = sc_F3;
    scancodes[RAWKEY_F4]              = sc_F4;
    scancodes[RAWKEY_F5]              = sc_F5;
    scancodes[RAWKEY_F6]              = sc_F6;
    scancodes[RAWKEY_F7]              = sc_F7;
    scancodes[RAWKEY_F8]              = sc_F8;
    scancodes[RAWKEY_F9]              = sc_F9;
    scancodes[RAWKEY_F10]             = sc_F10;
    scancodes[RAWKEY_F11]             = sc_F11;
    scancodes[RAWKEY_F12]             = sc_F12;

    scancodes[RAWKEY_TILDE]           = sc_Quote;
    scancodes[RAWKEY_1]               = sc_1;
    scancodes[RAWKEY_2]               = sc_2;
    scancodes[RAWKEY_3]               = sc_3;
    scancodes[RAWKEY_4]               = sc_4;
    scancodes[RAWKEY_5]               = sc_5;
    scancodes[RAWKEY_6]               = sc_6;
    scancodes[RAWKEY_7]               = sc_7;
    scancodes[RAWKEY_8]               = sc_8;
    scancodes[RAWKEY_9]               = sc_9;
    scancodes[RAWKEY_0]               = sc_0;
    scancodes[RAWKEY_MINUS]           = sc_Minus;
    scancodes[RAWKEY_EQUAL]           = sc_Equals;
    scancodes[RAWKEY_BACKSLASH]       = sc_BackSlash;
    scancodes[RAWKEY_BACKSPACE]       = sc_BackSpace;

    scancodes[RAWKEY_TAB]             = sc_Tab;
    scancodes[RAWKEY_Q]               = sc_Q;
    scancodes[RAWKEY_W]               = sc_W;
    scancodes[RAWKEY_E]               = sc_E;
    scancodes[RAWKEY_R]               = sc_R;
    scancodes[RAWKEY_T]               = sc_T;
    scancodes[RAWKEY_Y]               = sc_Y;
    scancodes[RAWKEY_U]               = sc_U;
    scancodes[RAWKEY_I]               = sc_I;
    scancodes[RAWKEY_O]               = sc_O;
    scancodes[RAWKEY_P]               = sc_P;
    scancodes[RAWKEY_LBRACKET]        = sc_OpenBracket;
    scancodes[RAWKEY_RBRACKET]        = sc_CloseBracket;
    scancodes[RAWKEY_RETURN]          = sc_Return;

    scancodes[RAWKEY_CONTROL]         = sc_LeftControl;
    scancodes[RAWKEY_CAPSLOCK]        = sc_CapsLock;
    scancodes[RAWKEY_A]               = sc_A;
    scancodes[RAWKEY_S]               = sc_S;
    scancodes[RAWKEY_D]               = sc_D;
    scancodes[RAWKEY_F]               = sc_F;
    scancodes[RAWKEY_G]               = sc_G;
    scancodes[RAWKEY_H]               = sc_H;
    scancodes[RAWKEY_J]               = sc_J;
    scancodes[RAWKEY_K]               = sc_K;
    scancodes[RAWKEY_L]               = sc_L;
    scancodes[RAWKEY_SEMICOLON]       = sc_SemiColon;

    scancodes[RAWKEY_LSHIFT]          = sc_LeftShift;
    scancodes[RAWKEY_Z]               = sc_Z;
    scancodes[RAWKEY_X]               = sc_X;
    scancodes[RAWKEY_C]               = sc_C;
    scancodes[RAWKEY_V]               = sc_V;
    scancodes[RAWKEY_B]               = sc_B;
    scancodes[RAWKEY_N]               = sc_N;
    scancodes[RAWKEY_M]               = sc_M;
    scancodes[RAWKEY_COMMA]           = sc_Comma;
    scancodes[RAWKEY_PERIOD]          = sc_Period;
    scancodes[RAWKEY_SLASH]           = sc_Slash;
    scancodes[RAWKEY_RSHIFT]          = sc_RightShift;

    scancodes[RAWKEY_LALT]            = sc_LeftAlt;
    scancodes[RAWKEY_SPACE]           = sc_Space;
    scancodes[RAWKEY_RALT]            = sc_RightAlt;


    scancodes[RAWKEY_DELETE]          = sc_Delete;

    scancodes[RAWKEY_UP]              = sc_UpArrow;
    scancodes[RAWKEY_LEFT]            = sc_LeftArrow;
    scancodes[RAWKEY_RIGHT]           = sc_RightArrow;
    scancodes[RAWKEY_DOWN]            = sc_DownArrow;


    scancodes[RAWKEY_KP_9]            = sc_kpad_9;
    scancodes[RAWKEY_KP_8]            = sc_kpad_8;
    scancodes[RAWKEY_KP_7]            = sc_kpad_7;
    scancodes[RAWKEY_KP_6]            = sc_kpad_6;
    scancodes[RAWKEY_KP_5]            = sc_kpad_5;
    scancodes[RAWKEY_KP_4]            = sc_kpad_4;
    scancodes[RAWKEY_KP_3]            = sc_kpad_3;
    scancodes[RAWKEY_KP_2]            = sc_kpad_2;
    scancodes[RAWKEY_KP_1]            = sc_kpad_1;
    scancodes[RAWKEY_KP_0]            = sc_kpad_0;
    scancodes[RAWKEY_KP_DECIMAL]      = sc_kpad_Period;
    scancodes[RAWKEY_KP_ENTER]        = sc_kpad_Enter;



    if (monitorType == PAL_MONITOR_ID) {
        // Note - Not a real PAL mode (will use 320x256).
        videoModes[0].w = 320;
        videoModes[0].h = 200;

        // Note - Not a real PAL mode (will use 320x256).
        videoModes[1].w = 320;
        videoModes[1].h = 240;

        videoModes[2].w = 320;
        videoModes[2].h = 256;

        videoModes[3].w = 640;
        videoModes[3].h = 480;
    } else {
        videoModes[0].w = 320;
        videoModes[0].h = 200;

        videoModes[1].w = 320;
        videoModes[1].h = 400;

        videoModes[2].w = 640;
        videoModes[2].h = 200;

        videoModes[3].w = 640;
        videoModes[3].h = 400;
    }


    // Timer stuff.
    timerinthandle = NULL;

    AMIGA_StartTimer(1000 / PLATFORM_TIMER_HZ);
    

    // All is cool.
    return 1;    
}

void _platform_shutdown(void)
{   
    AMIGA_CloseDisplay();
    
    AMIGA_EndTimer();
    

    if (workbenchClosed) {
        AMIGA_LogMessage("Opening the Workbench\n");
        OpenWorkBench();
        workbenchClosed = 0;
    }
    
    AMIGA_LogShutdown();
}

