#include <stdlib.h>
#include <string.h>

#include "dsl.h"
#include "util.h"

#if (defined PLATFORM_MACOSX)
	#include <SDL/SDL.h>
	#include <SDL_mixer/SDL_mixer.h>
#else
	#include "SDL.h"
	#include "SDL_mixer.h"
#endif

extern  int MV_MixPage;

static int DSL_Amiga_ErrorCode = DSL_Ok;

static int mixer_initialized;

static void ( *_CallBackFunc )( void );
static  char *_BufferStart;
static int _BufferSize;
static int _NumDivisions;
static int _SampleRate;
static int _remainder;

static Mix_Chunk *blank;
static unsigned char *blank_buf;

/*
possible todo ideas: cache sdl/sdl mixer error messages.
*/

char *DSL_Amiga_ErrorString( int Amiga_ErrorNumber )
{
	char *Amiga_ErrorString;
	
	switch (Amiga_ErrorNumber) {
		case DSL_Warning:
		case DSL_Amiga_Error:
			Amiga_ErrorString = DSL_Amiga_ErrorString(DSL_Amiga_ErrorCode);
			break;
		
		case DSL_Ok:
			Amiga_ErrorString = "SDL Driver ok.";
			break;
		
		case DSL_SDLInitFailure:
			Amiga_ErrorString = "SDL Audio initialization failed.";
			break;
		
		case DSL_MixerActive:
			Amiga_ErrorString = "SDL Mixer already initialized.";
			break;	
	
		case DSL_MixerInitFailure:
			Amiga_ErrorString = "SDL Mixer initialization failed.";
			break;
			
		default:
			Amiga_ErrorString = "Unknown SDL Driver error.";
			break;
	}
	
	return Amiga_ErrorString;
}

static void DSL_SetAmiga_ErrorCode(int Amiga_ErrorCode)
{
	DSL_Amiga_ErrorCode = Amiga_ErrorCode;
}

int DSL_Init( void )
{
	DSL_SetAmiga_ErrorCode(DSL_Ok);
	
	if (SDL_InitSubSystem(SDL_INIT_AUDIO) < 0) {
		DSL_SetAmiga_ErrorCode(DSL_SDLInitFailure);
		
		return DSL_Amiga_Error;
	}
	
	return DSL_Ok;
}

void DSL_Shutdown( void )
{
	DSL_StopPlayback();
}

static void mixer_callback(int chan, void *stream, int len, void *udata)
{
	Uint8 *stptr;
	Uint8 *fxptr;
	int copysize;
	
	/* len should equal _BufferSize, else this is screwed up */

	stptr = (Uint8 *)stream;
	
	if (_remainder > 0) {
		copysize = min(len, _remainder);
		
		fxptr = (Uint8 *)(&_BufferStart[MV_MixPage * 
			_BufferSize]);
		
		memcpy(stptr, fxptr+(_BufferSize-_remainder), copysize);
		
		len -= copysize;
		_remainder -= copysize;
		
		stptr += copysize;
	}
	
	while (len > 0) {
		/* new buffer */
		
		_CallBackFunc();
		
		fxptr = (Uint8 *)(&_BufferStart[MV_MixPage * 
			_BufferSize]);

		copysize = min(len, _BufferSize);
		
		memcpy(stptr, fxptr, copysize);
		
		len -= copysize;
		
		stptr += copysize;
	}
	
	_remainder = len;
}

int   DSL_BeginBufferedPlayback( char *BufferStart,
      int BufferSize, int NumDivisions, unsigned SampleRate,
      int MixMode, void ( *CallBackFunc )( void ) )
{
	Uint16 format;
	Uint8 *tmp;
	int channels;
	int chunksize;
		
	if (mixer_initialized) {
		DSL_SetAmiga_ErrorCode(DSL_MixerActive);
		
		return DSL_Amiga_Error;
	}
	
	_CallBackFunc = CallBackFunc;
	_BufferStart = BufferStart;
	_BufferSize = (BufferSize / NumDivisions);
	_NumDivisions = NumDivisions;
	_SampleRate = SampleRate;

	_remainder = 0;
	
	format = (MixMode & SIXTEEN_BIT) ? AUDIO_S16SYS : AUDIO_U8;
	channels = (MixMode & STEREO) ? 2 : 1;

/*
	23ms is typically ideal (11025,22050,44100)
	46ms isn't bad
*/
	
	chunksize = 512;
	
	if (SampleRate >= 16000) chunksize *= 2;
	if (SampleRate >= 32000) chunksize *= 2;
	
/*	
// SDL mixer does this already
	if (MixMode & SIXTEEN_BIT) chunksize *= 2;
	if (MixMode & STEREO) chunksize *= 2;
*/
	
	if (Mix_OpenAudio(SampleRate, format, channels, chunksize) < 0) {
		DSL_SetAmiga_ErrorCode(DSL_MixerInitFailure);
		
		return DSL_Amiga_Error;
	}

/*
	Mix_SetPostMix(mixer_callback, NULL);
*/
	/* have to use a channel because postmix will overwrite the music... */
	Mix_RegisterEffect(0, mixer_callback, NULL, NULL);
	
	/* create a dummy sample just to allocate that channel */
	blank_buf = (Uint8 *)malloc(4096);
	memset(blank_buf, 0, 4096);
	
	blank = Mix_QuickLoad_RAW(blank_buf, 4096);
		
	Mix_PlayChannel(0, blank, -1);
	
	mixer_initialized = 1;
	
	return DSL_Ok;
}

void DSL_StopPlayback( void )
{
	if (mixer_initialized) {
		Mix_HaltChannel(0);
	}
	
	if (blank != NULL) {
		Mix_FreeChunk(blank);
	}
	
	blank = NULL;
	
	if (blank_buf  != NULL) {
		free(blank_buf);
	}
	
	blank_buf = NULL;
	
	if (mixer_initialized) {
		Mix_CloseAudio();
	}
	
	mixer_initialized = 0;
}

unsigned DSL_GetPlaybackRate( void )
{
	return _SampleRate;
}

unsigned long DisableInterrupts( void )
{
	return 0;
}

void RestoreInterrupts( unsigned long flags )
{
}
